package com.equo.middleware.cef.provider;

import java.net.FileNameMap;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.eclipse.core.internal.boot.PlatformURLHandler;
import org.eclipse.osgi.storage.url.BundleResourceHandler;
import org.osgi.framework.BundleContext;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.ServiceReference;

import com.equo.chromium.swt.internal.spi.SchemeDomainPair;
import com.equo.chromium.swt.internal.spi.SchemeHandler;
import com.equo.chromium.swt.internal.spi.SchemeHandlerManager;
import com.equo.middleware.api.IMiddlewareService;
import com.equo.middleware.api.handler.IResponseHandler;
import com.equo.middleware.api.handler.ResourceHandler;
import com.equo.middleware.cef.provider.handler.DefaultSchemesResponseHandler;
import com.equo.middleware.cef.provider.handler.DelegatingSchemeHandler;
import com.equo.middleware.cef.provider.handler.PlatformSchemeResponseHandler;
import com.equo.middleware.cef.provider.handler.RejectingSchemeHandler;
import com.equo.middleware.cef.provider.media.WebCommonsFileNameMap;
import com.equo.middleware.provider.MiddlewareCollectionManager;
import com.equo.middleware.provider.MiddlewareServiceImpl;

/**
 * Implementation of the SchemeHandlerManager service.
 */
@SuppressWarnings("restriction")
public class CustomSchemeHandlerManager implements SchemeHandlerManager {

  private static final String ERROR_PREFIX = "CustomSchemeHandlerManager";

  private static final SchemeHandler REJECT_SCHEME_HANDLER = new RejectingSchemeHandler();

  private MiddlewareServiceImpl middlewareService;
  private MiddlewareCollectionManager middlewareCollectionManager;

  /**
   * Default constructor, initializes the needed services or logs an error.
   */
  public CustomSchemeHandlerManager() {
    IMiddlewareService svc = this.<IMiddlewareService>getService(IMiddlewareService.class);
    if (svc instanceof MiddlewareServiceImpl) {
      this.middlewareService = (MiddlewareServiceImpl) svc;
      middlewareCollectionManager =
          this.<MiddlewareCollectionManager>getService(MiddlewareCollectionManager.class);
      replaceCurrentFileNameMap();
    } else {
      System.err.printf("%s: %s. Expected: %s but found %s\n", ERROR_PREFIX,
          "Incompatible middleware service found", CustomSchemeHandlerManager.class.getName(),
          svc.getClass().getName());
    }
  }

  @Override
  public List<SchemeDomainPair> getRegisteredSchemes() {
    List<SchemeDomainPair> schemeDomainData = new ArrayList<>();
    if (middlewareService != null) {
      for (ResourceHandler resourceHandler : middlewareCollectionManager.getResourceHandlers()
          .values()) {
        schemeDomainData.add(
            SchemeDomainPair.of(resourceHandler.getSchemeName(), resourceHandler.getDomainName()));
      }
      addDefaultSchemeHandlers(schemeDomainData);
    }
    return schemeDomainData;
  }

  @Override
  public SchemeHandler getSchemeHandler(String scheme, String domain) {
    ResourceHandler resourceHandler =
        middlewareCollectionManager.getResourceHandlers().get(scheme + domain);
    if (resourceHandler != null) {
      final IResponseHandler responseHandler = resourceHandler.getResponseHandler();
      if (responseHandler != null) {
        return new DelegatingSchemeHandler(responseHandler);
      }
    }
    if (middlewareService.shouldBlockByDefault()) {
      for (String allowedDomain : middlewareCollectionManager.getAllowedDomains()) {
        if (allowedDomain.equals(domain)) {
          return null;
        }
      }
      return REJECT_SCHEME_HANDLER;
    }
    return null;
  }

  private void addDefaultSchemeHandlers(List<SchemeDomainPair> schemeDomainData) {
    Map<String, ResourceHandler> resourceHandlerMap =
        middlewareCollectionManager.getResourceHandlers();
    try {
      SchemeDomainPair[] defaultSchemes =
          { SchemeDomainPair.of(BundleResourceHandler.OSGI_ENTRY_URL_PROTOCOL, ""),
              SchemeDomainPair.of(BundleResourceHandler.OSGI_RESOURCE_URL_PROTOCOL, "") };
      DefaultSchemesResponseHandler handler = new DefaultSchemesResponseHandler();
      for (SchemeDomainPair scheme : defaultSchemes) {
        if (!resourceHandlerMap.containsKey(scheme.getScheme())) {
          schemeDomainData.add(scheme);
          middlewareService.addResourceHandler(scheme.getScheme(), "", handler);
        }
      }

    } catch (Exception e) {
      // If the bundles are not accessible we should not throw any exceptions.
    }
    try {
      SchemeDomainPair platform = SchemeDomainPair.of(PlatformURLHandler.PROTOCOL, "");
      schemeDomainData.add(platform);
      middlewareService.addResourceHandler(platform.getScheme(), "",
          new PlatformSchemeResponseHandler());
    } catch (Exception e) {
      // If the bundles are not accessible we should not throw any exceptions.
    }

    String[] standardSchemes = { "https", "http" };
    for (String scheme : standardSchemes) {
      if (!resourceHandlerMap.containsKey(scheme)) {
        schemeDomainData.add(SchemeDomainPair.of(scheme, ""));
      }
    }
  }

  private void replaceCurrentFileNameMap() {
    FileNameMap previous = URLConnection.getFileNameMap();
    URLConnection.setFileNameMap(new WebCommonsFileNameMap(previous));
  }

  private <T> T getService(Class<T> serviceClazz) {
    BundleContext ctx = FrameworkUtil.getBundle(getClass()).getBundleContext();
    if (ctx != null) {
      ServiceReference<T> serviceReference =
          (ServiceReference<T>) ctx.getServiceReference(serviceClazz);
      if (serviceReference != null) {
        T service = ctx.getService(serviceReference);
        return (T) service;
      } else {
        System.err.printf("%s: %s %s.", ERROR_PREFIX, "Couldn't find service of type",
            serviceClazz.getName());
      }
    }

    return null;
  }

}
